extends Node2D

# --- State Machine ---
enum State { MENU, COUNTDOWN, PLAYING, DYING, SCORE }
var current_state = State.MENU

# --- Game Data ---
var score = 0
var pipe_scene = preload("res://prefabs/pipe_pair.tscn")

# --- Gap Settings ---
const PIPE_GAP = 80       # Vertical gap between top and bottom pipes
const PIPE_MARGIN = 20    # Min distance from viewport top / ground
const PIPE_Y_MIN = PIPE_MARGIN + PIPE_GAP / 2 - 144   # -84
const PIPE_Y_MAX = 280 - PIPE_MARGIN - PIPE_GAP / 2 - 144  # 76
const MAX_PIPE_SHIFT = 60 # Max Y change between consecutive pipes

var last_pipe_y = 0.0

func _ready():
	$Bird.area_entered.connect(_on_bird_area_entered)
	$PipeSpawnTimer.timeout.connect(_on_pipe_spawn_timer_timeout)
	$Sound/Music.play()
	show_menu()

# === TODO 6: State Machine ===
# In _process(delta), use if/elif to check current_state:
#   State.MENU: if "flap" just pressed -> call start_countdown()
#   State.COUNTDOWN: (countdown runs itself via await)
#   State.PLAYING: (bird and pipes update themselves)
#   State.DYING: (freeze runs itself via await)
#   State.SCORE: if "flap" just pressed -> call restart()
func _process(_delta):
	# === TODO 6: State Machine ===
	if current_state == State.MENU:
		if Input.is_action_just_pressed("flap"):
			start_countdown()
	elif current_state == State.COUNTDOWN:
		pass  # Countdown runs itself via await
	elif current_state == State.PLAYING:
		pass  # Bird and pipes update themselves
	elif current_state == State.DYING:
		pass  # Freeze runs itself via await
	elif current_state == State.SCORE:
		if Input.is_action_just_pressed("flap"):
			restart()

# === TODO 4: Pipe Spawning ===
# Called when PipeSpawnTimer fires
# Create a new pipe_scene instance, set gap, place at x=550
# Clamp Y so the gap stays within viewport margins
# Limit Y shift from last pipe for a smooth path
# Add it to the scene with add_child()
# Add the pipe to the "pipe_pairs" group
func _on_pipe_spawn_timer_timeout():
	# === TODO 4: Pipe Spawning ===
	var pipe = pipe_scene.instantiate()
	pipe.gap = PIPE_GAP
	pipe.position.x = 550
	var min_y = max(PIPE_Y_MIN, last_pipe_y - MAX_PIPE_SHIFT)
	var max_y = min(PIPE_Y_MAX, last_pipe_y + MAX_PIPE_SHIFT)
	pipe.position.y = randf_range(min_y, max_y)
	last_pipe_y = pipe.position.y
	pipe.add_to_group("pipe_pairs")
	add_child(pipe)

# === TODO 5: Collision Handling ===
# Called when the bird enters any Area2D
# If area is in group "pipe" -> call game_over()
# If area is in group "score_zone" -> increment score, update label, free the zone
func _on_bird_area_entered(area):
	# === TODO 5: Collision Handling ===
	if area.is_in_group("pipe"):
		game_over()
	elif area.is_in_group("score_zone"):
		score += 1
		$ScoreLabel.text = str(score)
		$Sound/ScoreSound.play()
		area.queue_free()

func start_countdown():
	current_state = State.COUNTDOWN
	$ScoreLabel.visible = true
	$ReadyLabel.visible = false
	$GameOverLabel.visible = false
	$Bird.visible = false
	$ScoreLabel["theme_override_font_sizes/font_size"] = 80
	$ScoreLabel.text = "3"
	$Sound/ScoreSound.play()
	await get_tree().create_timer(1.0).timeout
	$ScoreLabel.text = "2"
	$Sound/ScoreSound.play()
	await get_tree().create_timer(1.0).timeout
	$ScoreLabel.text = "1"
	$Sound/ScoreSound.play()
	await get_tree().create_timer(1.0).timeout
	$ScoreLabel["theme_override_font_sizes/font_size"] = 58
	$ScoreLabel.text = "0"
	$Bird.visible = true
	start_game()

func start_game():
	current_state = State.PLAYING
	score = 0
	$ScoreLabel.text = "0"
	$ScoreLabel.visible = true
	$Bird.alive = true
	$PipeSpawnTimer.start()

func game_over():
	current_state = State.DYING
	$Bird.die()
	$PipeSpawnTimer.stop()
	$Sound/HitSound.play()
	$Background.set_process(false)
	$Ground.set_process(false)
	get_tree().call_group("pipe_pairs", "set_process", false)
	await get_tree().create_timer(1.0).timeout
	show_score_screen()

func show_score_screen():
	current_state = State.SCORE
	$ScoreLabel.visible = false
	$ReadyLabel["theme_override_font_sizes/font_size"] = 40
	$ReadyLabel.offset_top = 60
	$ReadyLabel.text = "You Lost!"
	$ReadyLabel.visible = true
	$GameOverLabel["theme_override_font_sizes/font_size"] = 14
	$GameOverLabel.offset_top = 120
	$GameOverLabel.text = "Score: " + str(score) + "\nPress Space to play again!"
	$GameOverLabel.visible = true

func restart():
	$ReadyLabel.visible = false
	$GameOverLabel.visible = false
	get_tree().call_group("pipe_pairs", "queue_free")
	$Bird.reset()
	last_pipe_y = 0.0
	$Background.set_process(true)
	$Ground.set_process(true)
	show_menu()

func show_menu():
	current_state = State.MENU
	$Bird.alive = false
	$ReadyLabel["theme_override_font_sizes/font_size"] = 40
	$ReadyLabel.offset_top = 60
	$ReadyLabel.text = "Flappy Bird"
	$ReadyLabel.visible = true
	$GameOverLabel["theme_override_font_sizes/font_size"] = 14
	$GameOverLabel.offset_top = 120
	$GameOverLabel.text = "Press Space"
	$GameOverLabel.visible = true
	$ScoreLabel.text = ""
	score = 0
